module net.BurtonRadons.dig.platform.radioButton;

private import net.BurtonRadons.dig.platform.base;
private import net.BurtonRadons.dig.platform.control;

/** This control provides a text label with a button beside it that
  has on or off state - traditionally a circle with a black dot in it
  and an empty circle respectively.  These are organised into groups
  of similar types with only one selected.

  You assign the text label using the #caption and #formatCaption
  methods, and change whether the button is on with #enabled.  See the
  RadioGroup class for how you can more easily manage radio sets.

  These are the supported dispatchers:
  - #onClick: Sent when the radio button is clicked on.
  - #onDClick: Sent when the radio button is clicked on again in rapid succession (a double-click); if this is empty, #onClick is notified instead.

  */

class RadioButton : Control
{
    private import net.BurtonRadons.dig.platform.font;
    private import net.BurtonRadons.dig.platform.windows;
	private import net.BurtonRadons.dig.common.dispatcher;
    
    Dispatcher onClick; /**< Dispatched when the button is clicked upon. */
    Dispatcher onDClick; /**< Dispatched when the button is double-clicked upon; if this is empty, #onClick is used instead. */

    char [1] digPlatformBoundLetter; /**< The letter that this is bound to, or \0 for none. */

    /** Register with the parent and create the button. */
    this (Control parent)
    {
        super (parent);
        digPlatformStyle |= std.c.windows.windows.WS_VISIBLE | std.c.windows.windows.WS_CHILD | std.c.windows.windows.BS_NOTIFY | std.c.windows.windows.BS_RADIOBUTTON;
        digPlatformStyle &= ~(std.c.windows.windows.WS_HSCROLL | std.c.windows.windows.WS_VSCROLL);
        digPlatformHWNDCreate (0, "BUTTON", null, digPlatformStyle, (_HANDLE) 0);
    }

    /** Change the textual caption in the button.
      * Text where & is followed by a character is interpreted
      * specially to underline that character (but has no higher-level
      * effects).
      */

    void caption (char [] text)
    {
        if (digPlatformBoundLetter [0])
            bindings.remove (digPlatformBoundLetter, &onClick);

        digPlatformSetCaption (text);
        recalculate ();

        int offset = std.string.find (text, '&');

        if (offset >= 0)
        {
            digPlatformBoundLetter [0] = text [offset + 1];
            bind (digPlatformBoundLetter, &onClick);
        }
    }

    /** If set, the text inside the button is aligned to the left border; otherwise it is centered. */
    void alignLeft (bit value) { digPlatformSetStyle (std.c.windows.windows.BS_LEFT, value); }

    /** If set, the text inside the button is aligned to the right border; otherwise it is centered. */
    void alignRight (bit value) { digPlatformSetStyle (std.c.windows.windows.BS_RIGHT, value); }

    /** If set, the text inside the button is aligned to the top; otherwise it is vertically centered. */
    void alignTop (bit value) { digPlatformSetStyle (std.c.windows.windows.BS_TOP, value); }

    /** Return whether the radio button is checked. */
    bit checked ()
    {
        int value = SendMessageA (digPlatformHWND, std.c.windows.windows.BM_GETSTATE, 0, 0);

        return (bit) (value & std.c.windows.windows.BST_CHECKED);
    }

    /** Set whether the radio button is checked. */
    void checked (bit value)
    {
        SendMessageA (digPlatformHWND, std.c.windows.windows.BM_SETCHECK, value, 0);
    }

    /** Set whether this is active (true) or grayed and inactive (false). */
    void enabled (bit value)
    {
        EnableWindow (digPlatformHWND, value);
    }

    override void recalculate ()
    {
        _HGDIOBJ font;
        _DWORD result;

        font = (_HANDLE) SendMessageA (digPlatformHWND, WM_GETFONT, 0, 0);
        SelectObject (digPlatformHDC, font);
        result = GetTabbedTextExtentW (digPlatformHDC, digPlatformCaption, digPlatformCaption.length, 0, null);
        digPlatformSuggestWidth = (result & 0xFFFF) + 8;
        digPlatformSuggestHeight = result >> 16;
        digPlatformBorder (digPlatformSuggestWidth, digPlatformSuggestHeight);
        digPlatformMoved ();
    }

/+
#ifndef DOXYGEN_SHOULD_SKIP_THIS
+/

    override void digPlatformCommand (int code, int id)
    {
        switch (code)
        {
            case std.c.windows.windows.BN_CLICKED: onClick.notify (); break;
            case std.c.windows.windows.BN_DBLCLK: onClick.notifyOrEmpty (&onDClick); break;
            case std.c.windows.windows.BN_KILLFOCUS: break;
            case std.c.windows.windows.BN_SETFOCUS: break;
            default:
                printf ("not handled: %d %d\n", code, id);
        }
    }

    void digPlatformBorder (inout int width, inout int height)
    {
        width = width + 6;
        height = height + 6;
    }

/+
#endif
+/
}
